/*
 *robotfindskitten: A Zen simulation
 *
 *Copyright (C) 1997,2000 Leonard Richardson
 *                        leonardr@segfault.org
 *                        http://www.crummy.com/devel/
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or EXISTANCE OF KITTEN.  See the GNU General
 *   Public License for more details.
 *
 *   http://www.gnu.org/copyleft/gpl.html
 *
 *ported to the Wii by chris 19/10/2009
 */

#include <stdlib.h>
#include <unistd.h>
#include <gccore.h>
#include <wiiuse/wpad.h>

/*The messages go in a separate file because they are collectively
  huge, and you might want to modify them. It would be nice to load
  the messages from a text file at run time.*/
#include "messages.h"

#define ver "1.7320508.406/wii::Cheshire"
#define POLL_DELAY 300000 /*microseconds between polling joysticks and
                           balance board */
#define SENSITIVITY 15 /*balance board sensitivity*/

/*Constants for our internal representation of the screen.*/
#define EMPTY -1
#define ROBOT 0
#define KITTEN 1

/*Screen dimensions.*/
#define X_MIN 2
#define X_MAX (COLS - 3)
#define Y_MIN 5
#define Y_MAX (LINES - 1)

/*Macros for generating numbers in different ranges*/
#define randx() rand() % (X_MAX-X_MIN+1)+X_MIN
#define randy() rand() % (Y_MAX-Y_MIN+1)+Y_MIN /*I'm feeling randy()!*/
#define randchar() rand() % (126-'!'+1)+'!';
#define randcolor() rand() % 6 + 1
#define randbold() (rand() % 2 ? TRUE:FALSE)

/*Row constants for the animation*/
#define ADV_ROW 3
#define ANIMATION_MEET 50

/*This struct contains all the information we need to display an object
  on the screen*/
typedef struct
{
  int x;
  int y;
  int color;
  bool bold;
  char character;
} screen_object;

/*
 *Function definitions
 */

/*Initialization and setup functions*/
void initialize_wii();
void initialize_arrays();
void initialize_robot();
void initialize_kitten();
void initialize_bogus();
void initialize_screen();
void instructions();
void title_pic();

/*Game functions*/
void play_game();
void process_input(int);

/*Helper functions*/
int validchar(char);

void play_animation(int);
int board_check();

/*Global variables. Bite me, it's fun.*/
screen_object robot;
screen_object kitten;

int num_bogus;
screen_object bogus[MESSAGES];
int bogus_messages[MESSAGES];
int used_messages[MESSAGES];

char power=0; /*Wii power buttons*/

/* This array contains our internal representation of the screen. The
 array is bigger than it needs to be, as we don't need to keep track
 of the first few rows of the screen. But that requires making an
 offset function and using that everywhere. So not right now. */
int** screen = NULL;

#include "wiicurses.h"
#include "draw.h"

/******************************************************************************
 *
 * Begin meaty routines that do the dirty work.
 *
 *****************************************************************************/

/*
 *play_game waits in a loop getting input and sending it to process_input
 */
void play_game()
{
  int old_x = robot.x;
  int old_y = robot.y;
  int input=0;
  short gcinput=0;
  char delay=0;
  struct expansion_t data, board;
  board.type=WPAD_EXP_WIIBOARD;
  while (!((input&WPAD_BUTTON_HOME)||(gcinput&PAD_BUTTON_START)))
    {
      process_input(input);

      /*Redraw robot, where applicable. We're your station, robot.*/
      if (!(old_x == robot.x && old_y == robot.y))
	{
	  /*Get rid of the old robot*/
	  mvaddch(old_y,old_x, ' ');
	  screen[old_x][old_y] = EMPTY;

	  /*Meet the new robot, same as the old robot.*/
	  draw(robot);
	  refresh();
	  screen[robot.x][robot.y] = ROBOT;

	  old_x = robot.x;
	  old_y = robot.y;
	}
	if(delay)
	{
		usleep(POLL_DELAY);
		delay=0;
	}
	PAD_ScanPads();
	gcinput=PAD_ButtonsDown(0);
	/*1.Wiimote*/
    WPAD_ScanPads();
    input = WPAD_ButtonsDown(0);
    WPAD_Expansion(WPAD_CHAN_0,&data);
    /*2.Nunchuck*/
   	if(data.type==WPAD_EXP_NUNCHUK)
   	{
		if((data.nunchuk.js.pos.x-data.nunchuk.js.center.x)<-60)
		{
			input|=WPAD_BUTTON_LEFT;
			delay=1;
		}
		else if((data.nunchuk.js.pos.x-data.nunchuk.js.center.x)>60)
		{
			input|=WPAD_BUTTON_RIGHT;
			delay=1;
		}
		if((data.nunchuk.js.pos.y-data.nunchuk.js.center.y)<-60)
		{
			input|=WPAD_BUTTON_DOWN;
			delay=1;
		}
		else if((data.nunchuk.js.pos.y-data.nunchuk.js.center.y)>60)
		{
			input|=WPAD_BUTTON_UP;
			delay=1;
		}
	}
	/*3.Classic Controller.  I haven't tested this; I don't have one*/
	else if(data.type==WPAD_EXP_CLASSIC)
	{
		if((data.classic.rjs.pos.x-data.classic.rjs.center.x<-60)||(data.classic.ljs.pos.x-data.classic.ljs.center.x<-60))
		{
			input|=WPAD_BUTTON_LEFT;
			delay=1;
		}
		else if((data.classic.rjs.pos.x-data.classic.rjs.center.x>60)||(data.classic.ljs.pos.x-data.classic.ljs.center.x>60))
		{
			input|=WPAD_BUTTON_RIGHT;
			delay=1;
		}
		if((data.classic.rjs.pos.y-data.classic.rjs.center.y<-60)||(data.classic.ljs.pos.y-data.classic.ljs.center.y<-60))
		{
			input|=WPAD_BUTTON_DOWN;
			delay=1;
		}
		else if((data.classic.rjs.pos.y-data.classic.rjs.center.y>60)||(data.classic.ljs.pos.y-data.classic.ljs.center.y>60))
		{
			input|=WPAD_BUTTON_UP;
			delay=1;
		}

	}
	/*4.Gamecube controller*/
	if(gcinput&PAD_BUTTON_LEFT||PAD_StickX(0)<-50||PAD_SubStickX(0)<-50)
	{
		input|=WPAD_BUTTON_LEFT;
		delay=1;
	}
	else if(gcinput&PAD_BUTTON_RIGHT||PAD_StickX(0)>50||PAD_SubStickX(0)>50)
	{
		input|=WPAD_BUTTON_RIGHT;
		delay=1;
	}
	if(gcinput&PAD_BUTTON_DOWN||PAD_StickY(0)<-50||PAD_SubStickY(0)<-50)
	{
		input|=WPAD_BUTTON_DOWN;
		delay=1;
	}
	else if(gcinput&PAD_BUTTON_UP||PAD_StickY(0)>50||PAD_SubStickY(0)>50)
	{
		input|=WPAD_BUTTON_UP;
		delay=1;
	}
	/*5.Balance Board*/
	if(board_check())
	{
		WPAD_Expansion(WPAD_BALANCE_BOARD,&board);
		if(board.wb.x<-SENSITIVITY)
		{
			input|=WPAD_BUTTON_LEFT;
			delay=1;
		}
		else if(board.wb.x>SENSITIVITY)
		{
			input|=WPAD_BUTTON_RIGHT;
			delay=1;
		}
		if(board.wb.y<-SENSITIVITY)
		{
			input|=WPAD_BUTTON_UP;
			delay=1;
		}
		else if(board.wb.y>SENSITIVITY)
		{
			input|=WPAD_BUTTON_DOWN;
			delay=1;
		}
	}

    if(power==1)
	{
		SYS_ResetSystem(SYS_POWEROFF,0,0);
		exit(0);
	}
	else if(power==2)
		SYS_ResetSystem(SYS_RETURNTOMENU,0,0);


  }
  message("Bye!");
  refresh();
  exit(0);
}

/*
 *Given the keyboard input, process_input interprets it in terms of moving,
 *touching objects, etc.
 */
void process_input(int input)
{
  int check_x = robot.x;
  int check_y = robot.y;

  if((input&WPAD_BUTTON_UP)||(input&CLASSIC_CTRL_BUTTON_UP))
    check_y--;
  if((input&WPAD_BUTTON_DOWN)||(input&CLASSIC_CTRL_BUTTON_DOWN))
    check_y++;
  if((input&WPAD_BUTTON_LEFT)||(input&CLASSIC_CTRL_BUTTON_LEFT))
    check_x--;
  if((input&WPAD_BUTTON_RIGHT)||(input&CLASSIC_CTRL_BUTTON_RIGHT))
    check_x++;

  /*Check for going off the edge of the screen.*/
  if (check_y < Y_MIN || check_y > Y_MAX || check_x < X_MIN || check_x > X_MAX)
    {
      return; /*Do nothing.*/
    }

  /*Check for collision*/
  if (screen[check_x][check_y] != EMPTY)
    {
      switch (screen[check_x][check_y])
	{
	case ROBOT:
				/*We didn't move, or we're stuck in a
				  time warp or something.*/
	  break;
	case KITTEN: /*Found it!*/
	  move(3,0);
	  clrtoeol();
	  play_animation(input);
	  break;
	default: /*We hit a bogus object; print its message.*/
	  message(messages[bogus_messages[screen[check_x][check_y]-2]]);
	  break;
	}
      return;
    }

  /*Otherwise, move the robot.*/
  robot.x = check_x;
  robot.y = check_y;
}

/******************************************************************************
 *
 * Begin helper routines
 *
 *****************************************************************************/

int validchar(char a)
{
  switch(a)
    {
    case '#':
    case ' ':
    case 127:
      return 0;
    }
  return 1;
}

void play_animation(int input)
{
  int counter;
  /*The grand cinema scene.*/
  WPAD_Rumble(0,1);
  PAD_ControlMotor(0,1);
  for (counter = 4; counter >0; counter--)
    {
      /*Move the object on the right.*/
      mvaddch(ADV_ROW,ANIMATION_MEET+counter+1,' ');
      move(ADV_ROW,ANIMATION_MEET+counter);
      if ((input&WPAD_BUTTON_RIGHT)||(input&WPAD_BUTTON_DOWN))
	draw_in_place(kitten);
      else
	draw_in_place(robot);

      /*Move the object on the left.*/
      mvaddch(ADV_ROW,ANIMATION_MEET-counter,' ');
      move(ADV_ROW,ANIMATION_MEET-counter+1);
      if ((input&WPAD_BUTTON_RIGHT)||(input&WPAD_BUTTON_DOWN))
	draw_in_place(robot);
      else
	draw_in_place(kitten);
    refresh();
    sleep (1);
    WPAD_Rumble(0,0);
    PAD_ControlMotor(0,0);
    WPAD_Rumble(0,1);
    PAD_ControlMotor(0,1);
    }

  move(3,2);
  addstr("You found kitten! Way to go, robot!");
  WPAD_Rumble(0,0);
  PAD_ControlMotor(0,0);
  sleep(3);
  clear();
  initialize_arrays();
  initialize_robot();
  initialize_kitten();
  initialize_bogus();
  instructions();
  initialize_screen();
  play_game();
}

int board_check()
{
	unsigned int device;
	WPAD_Probe(WPAD_BALANCE_BOARD,&device);
	if(device==WPAD_EXP_WIIBOARD)
		return 1;
	else
		return 0;
}

/******************************************************************************
 *
 * Begin initialization routines (called before play begins).
 *
 *****************************************************************************/

void powerbutton1(){
	power=1;}
void powerbutton2(int channel){
	power=1;}
void resetbutton(){
	power=2;}

void title_pic() /*nicked off robotfindskitten.org*/
{
	printw("\n\n");
	printw("\x1b[36;1m              [");
	printw("\x1b[33;1m-");
	printw("\x1b[36;1m]                ");
	printw("\x1b[31;1m.::. .::.              ");
	printw("\x1b[33;0m|\\_/|    \n");
	printw("\x1b[36;1m              (");
	printw("\x1b[37;0m+");
	printw("\x1b[36;1m)");
	printw("\x1b[32;0m=C              ");
	printw("\x1b[31;1m:::::::::              ");
	printw("\x1b[33;0m|");
	printw("\x1b[32;1mo o");
	printw("\x1b[33;1m|__  \n");
	printw("\x1b[36;1m              | |                ");
	printw("\x1b[31;1m`:::::::'              ");
	printw("\x1b[37;1m=-");
	printw("\x1b[35;0m*");
	printw("\x1b[37;1m-=");
	printw("\x1b[33;0m__\\ \n");
	printw("\x1b[33;1m              OOO                  ");
	printw("\x1b[31;1m`:::'               ");
	printw("\x1b[33;0mc_c__(___)\n");
	printw("\x1b[37;1m\n\n");
}

void instructions()
{
  int input=0;
  short gcinput=0;
  PAD_ScanPads();
  WPAD_ScanPads();
  mvprintw(3,2,"    robotfindskitten v%s\n\n",ver);
  printw("    By the illustrious Leonard Richardson (C) 1997, 2000\n");
  printw("    Written originally for the Nerth Pork robotfindskitten contest\n\n");
  printw("    Wii port by chris, 21/10/2009  www.toxicbreakfast.com TOX 011\n\n");
  title_pic();
  printw("    In this game, you are robot (");
  draw_in_place(robot);
  printw("). Your job is to find kitten. This task\n");
  printw("    is complicated by the existence of various things which are not kitten.\n");
  printw("    Robot must touch items to determine if they are kitten or not. The game\n");
  printw("    ends when robotfindskitten. Alternatively, you may end the game by\n");
  printw("    hitting the Home key.\n\n\n");
  printw("                               Press A to start.\n");
  printw("                              Press HOME to exit.\n");
  refresh();

  while(!((input&WPAD_BUTTON_A)||(input&CLASSIC_CTRL_BUTTON_A)||(gcinput&PAD_BUTTON_A)))
  {
	PAD_ScanPads();
	gcinput=PAD_ButtonsDown(0);
    WPAD_ScanPads();
    input = WPAD_ButtonsDown(0);
    if((input&WPAD_BUTTON_HOME)||(gcinput&PAD_BUTTON_START))
    	exit(0);
    if(power==1)
	{
		SYS_ResetSystem(SYS_POWEROFF,0,0);
		exit(0);
	}
	else if(power==2)
		SYS_ResetSystem(SYS_RETURNTOMENU,0,0);
  }
  clear();
}

void initialize_arrays()
{
  int counter, counter2;
  screen_object empty;
  int i = 0;

  /* Allocate memory for the screen. */
  screen = malloc (sizeof (int*) * (X_MAX + 1));
  for (i = 0; i < (X_MAX + 1); ++i) {
          /* XXX: blah blah blah check for NULL */
          screen[i] = malloc (sizeof (int) * (Y_MAX + 1));
  }

  /*Initialize the empty object.*/
  empty.x = -1;
  empty.y = -1;
  empty.color = 0;
  empty.bold = FALSE;
  empty.character = ' ';

  for (counter = 0; counter <= X_MAX; counter++)
    {
      for (counter2 = 0; counter2 <= Y_MAX; counter2++)
	{
	  screen[counter][counter2] = EMPTY;
	}
    }

  /*Initialize the other arrays.*/
  for (counter = 0; counter < MESSAGES; counter++)
    {
      used_messages[counter] = 0;
      bogus_messages[counter] = 0;
      bogus[counter] = empty;
    }
}

/*Wii initialisation*/
void initialize_wii()
{
	static void *xfb = NULL;
	static GXRModeObj *rmode = NULL;
	VIDEO_Init();
	PAD_Init();
	WPAD_Init();
	rmode = VIDEO_GetPreferredMode(NULL);
	xfb = MEM_K0_TO_K1(SYS_AllocateFramebuffer(rmode));
	console_init(xfb,20,20,rmode->fbWidth,rmode->xfbHeight,rmode->fbWidth*VI_DISPLAY_PIX_SZ);
	VIDEO_Configure(rmode);
	VIDEO_SetNextFramebuffer(xfb);
	VIDEO_SetBlack(FALSE);
	VIDEO_Flush();
	VIDEO_WaitVSync();
	if(rmode->viTVMode&VI_NON_INTERLACE) VIDEO_WaitVSync();
	SYS_SetResetCallback(resetbutton);
	SYS_SetPowerCallback(powerbutton1);
	WPAD_SetPowerButtonCallback(powerbutton2);
}


/*initialize_robot initializes robot.*/
void initialize_robot()
{
  /*Assign a position to the player.*/
  robot.x = randx();
  robot.y = randy();

  robot.character = '#';
  robot.color = 7;
  robot.bold = FALSE;
  screen[robot.x][robot.y] = ROBOT;
}

/*initialize kitten, well, initializes kitten.*/
void initialize_kitten()
{
  /*Assign the kitten a unique position.*/
  do
    {
      kitten.x = randx();
      kitten.y = randy();
    } while (screen[kitten.x][kitten.y] != EMPTY);

  /*Assign the kitten a character and a color.*/
  do {
    kitten.character = randchar();
  } while (!(validchar(kitten.character)));
  screen[kitten.x][kitten.y] = KITTEN;

  kitten.color = randcolor();
  kitten.bold = randbold();
}

/*initialize_bogus initializes all non-kitten objects to be used in this run.*/
void initialize_bogus()
{
  int counter, index;
  for (counter = 0; counter < num_bogus; counter++)
    {
      /*Give it a color.*/
      bogus[counter].color = randcolor();
      bogus[counter].bold = randbold();

      /*Give it a character.*/
      do {
	bogus[counter].character = randchar();
      } while (!(validchar(bogus[counter].character)));

      /*Give it a position.*/
      do
	{
	  bogus[counter].x = randx();
	  bogus[counter].y = randy();
	} while (screen[bogus[counter].x][bogus[counter].y] != EMPTY);

      screen[bogus[counter].x][bogus[counter].y] = counter+2;

      /*Find a message for this object.*/
      do {
	index = rand() % MESSAGES;
      } while (used_messages[index] != 0);

      bogus_messages[counter] = index;
      used_messages[index] = 1;
    }

}

/*initialize_screen paints the screen.*/
void initialize_screen()
{
  int counter;

  /*
   *Print the status portion of the screen.
   */
  mvprintw(2,2,"\x1b[33;0mrobotfindskitten v%s\n\n",ver);
  printw("\x1b[37;1m");

  /*Draw a line across the screen.*/
  move(4,X_MIN);
  for (counter = X_MIN; counter <= X_MAX; counter++)
    {
      printw("%c",95);
    }

  /*
   *Draw all the objects on the playing field.
   */
  for (counter = 0; counter < num_bogus; counter++)
    {
      draw(bogus[counter]);
    }

  draw(kitten);
  draw(robot);

  refresh();

}

int main(int argc, char *argv[])
{
  num_bogus=20;
  /*Initialize the random number generator*/
  srand(time(0));

  /* Set up the screen to use the IBM character set. ncurses still won't
   cooperate with characters before '!', so we take care of that in the
   randchar() macro.
   printf("%c%c%c",27,'(','U');*/

  initialize_wii();

  initialize_arrays();

  /*
   *Now we initialize the various game objects.
   */
  initialize_robot();
  initialize_kitten();
  initialize_bogus();

  instructions();

  initialize_screen();
  play_game();
  return 0;
}
